# WindowsHelloAuth.ps1
# Real Windows Hello authentication using WinRT APIs

# C# helper class for Windows Hello authentication
$windowsHelloCode = @"
using System;
using System.Runtime.InteropServices;
using System.Threading.Tasks;

public class WindowsHelloAuth
{
    [DllImport("user32.dll")]
    private static extern IntPtr GetForegroundWindow();

    public static async Task<bool> RequestAuthenticationAsync(string message)
    {
        try
        {
            // Check if Windows Hello is available
            var availability = await Windows.Security.Credentials.UI.UserConsentVerifier.CheckAvailabilityAsync();

            if (availability != Windows.Security.Credentials.UI.UserConsentVerifierAvailability.Available)
            {
                Console.WriteLine("Windows Hello is not available on this device.");
                Console.WriteLine("Availability status: " + availability.ToString());
                return false;
            }

            // Request verification using Windows Hello
            var result = await Windows.Security.Credentials.UI.UserConsentVerifier.RequestVerificationAsync(message);

            return result == Windows.Security.Credentials.UI.UserConsentVerificationResult.Verified;
        }
        catch (Exception ex)
        {
            Console.WriteLine("Error during Windows Hello authentication: " + ex.Message);
            return false;
        }
    }

    public static bool RequestAuthentication(string message)
    {
        return RequestAuthenticationAsync(message).GetAwaiter().GetResult();
    }

    public static async Task<string> CheckAvailabilityAsync()
    {
        try
        {
            var availability = await Windows.Security.Credentials.UI.UserConsentVerifier.CheckAvailabilityAsync();
            return availability.ToString();
        }
        catch (Exception ex)
        {
            return "Error: " + ex.Message;
        }
    }

    public static string CheckAvailability()
    {
        return CheckAvailabilityAsync().GetAwaiter().GetResult();
    }
}
"@

function Test-WindowsHelloAvailable {
    try {
        # Try to load the Windows Runtime assemblies
        [Windows.Security.Credentials.UI.UserConsentVerifier, Windows.Security.Credentials.UI, ContentType=WindowsRuntime] | Out-Null

        # Compile C# helper class
        Add-Type -TypeDefinition $windowsHelloCode -ReferencedAssemblies @(
            'System.Runtime.WindowsRuntime',
            [Windows.Security.Credentials.UI.UserConsentVerifier].Assembly.Location
        ) -ErrorAction Stop

        # Check availability
        $availability = [WindowsHelloAuth]::CheckAvailability()

        if ($availability -eq "Available") {
            return $true
        } else {
            Write-Warning "Windows Hello status: $availability"
            return $false
        }
    }
    catch {
        Write-Warning "Windows Hello is not available: $_"
        return $false
    }
}

function Invoke-WindowsHelloAuth {
    param(
        [string]$Message = "Please authenticate to continue"
    )

    try {
        # Load WinRT types if not already loaded
        try {
            [Windows.Security.Credentials.UI.UserConsentVerifier, Windows.Security.Credentials.UI, ContentType=WindowsRuntime] | Out-Null
        } catch {
            throw "Failed to load Windows Runtime assemblies. Windows Hello may not be available on this system."
        }

        # Compile C# helper class if not already compiled
        if (-not ([System.Management.Automation.PSTypeName]'WindowsHelloAuth').Type) {
            Add-Type -TypeDefinition $windowsHelloCode -ReferencedAssemblies @(
                'System.Runtime.WindowsRuntime',
                [Windows.Security.Credentials.UI.UserConsentVerifier].Assembly.Location
            ) -ErrorAction Stop
        }

        # Request authentication
        Write-Host "Requesting Windows Hello authentication..." -ForegroundColor Cyan
        $result = [WindowsHelloAuth]::RequestAuthentication($Message)

        if ($result) {
            Write-Host "[OK] Windows Hello authentication successful" -ForegroundColor Green
            return $true
        } else {
            Write-Host "[ERROR] Windows Hello authentication failed or was cancelled" -ForegroundColor Red
            return $false
        }
    }
    catch {
        Write-Error "Windows Hello authentication error: $_"
        return $false
    }
}

Export-ModuleMember -Function @('Test-WindowsHelloAvailable', 'Invoke-WindowsHelloAuth')
